package gov.va.vamf.scheduling.direct.datalayer.clinic;

import com.agilex.healthcare.directscheduling.dao.DAOConstants;
import com.agilex.healthcare.directscheduling.dataservice.DSFacilityProviderDataService;
import com.agilex.healthcare.utility.NullChecker;
import gov.va.vamf.scheduling.direct.domain.CdwClinic;
import gov.va.vamf.scheduling.direct.domain.CdwClinics;
import gov.va.vamf.scheduling.direct.domain.ClinicalService;
import gov.va.vamf.scheduling.direct.domain.StopCode;
import gov.va.vamf.scheduling.direct.domain.StopCodes;
import org.springframework.jdbc.core.JdbcTemplate;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ClinicDaoImpl {

    private ClinicStoredProcedure clinicStoredProcedure;
    private JdbcTemplate jdbcTemplate;
    private String storedProcedure;
    private ClinicResultSetRowMapper clinicResultSetRowMapper;

    @Resource
    DSFacilityProviderDataService dsFacilityProviderDataService;

    public void initStoredProcedure() {
        clinicStoredProcedure = new ClinicStoredProcedure(jdbcTemplate, storedProcedure, clinicResultSetRowMapper);
    }

    public CdwClinics retrieveDisplayToPatientClinicsByStopCode(
            String institutionCode,
            String stopCode,
            Collection<StopCode> excludedStopCodes
    ) {
        String stopCodesAsString = ClinicalService.getStopCodesAsString(dsFacilityProviderDataService.createStopCodesFromPrimary(stopCode));

        CdwClinics cdwClinics = retrieveDirectSchedulingClinics(institutionCode.substring(0, 3), stopCodesAsString);

        return filterCdwClinics(cdwClinics, institutionCode, excludedStopCodes);
    }

    public CdwClinics retrieveDirectSchedulingClinics(String siteCode, String stopCodePairs) {

        Map<String, Object> inputParamsMap = new HashMap<>();
        inputParamsMap.put(DAOConstants.STATION, siteCode);
        inputParamsMap.put(DAOConstants.STOP_CODE_PAIRS, stopCodePairs);
        inputParamsMap.put(DAOConstants.DISP_APPT_FLAG, "Y");
        inputParamsMap.put(DAOConstants.DIRECT_SCHED_FLAG, "Y");
        inputParamsMap.put(DAOConstants.CLINIC_IENS, null);

        return executeStoredProcedure(inputParamsMap);
    }

    public CdwClinics retrieveDirectSchedulingClinicsByStopCodes(String siteCode, StopCodes stopCodes) {

        String stopCodePairs = ClinicalService.getStopCodesAsString(stopCodes);

        return retrieveDirectSchedulingClinics(siteCode, stopCodePairs);
    }

    private CdwClinics executeStoredProcedure(Map<String, Object> inputParamsMap) {
        Map<String, Object> storedProcedureResults = clinicStoredProcedure.getStoredProcedureResults(inputParamsMap);
        List<CdwClinic> clinicsList = (List<CdwClinic>) storedProcedureResults.get(DAOConstants.RESULT_LIST);

        CdwClinics cdwClinics = new CdwClinics();
        cdwClinics.addAll(clinicsList);

        return cdwClinics;
    }

    public CdwClinics retrieveDisplayToPatientClinics(String siteCode) {
        return retrieveDisplayToPatientClinics(siteCode, null);
    }

    public CdwClinics retrieveDisplayToPatientClinics(String siteCode, Collection<String> clinicIENs) {
        Map<String, Object> inputParamsMap = new HashMap<>();
        inputParamsMap.put(DAOConstants.STATION, siteCode);
        inputParamsMap.put(DAOConstants.STOP_CODE_PAIRS, null);
        inputParamsMap.put(DAOConstants.DISP_APPT_FLAG, "Y");
        inputParamsMap.put(DAOConstants.DIRECT_SCHED_FLAG, null);

        String clinicIensString = null;

        if (NullChecker.isNotNullish(clinicIENs) && !clinicIENs.isEmpty()) {
            clinicIensString = String.join(",", clinicIENs);
        }

        inputParamsMap.put(DAOConstants.CLINIC_IENS, clinicIensString);

        return executeStoredProcedure(inputParamsMap);
    }

    public CdwClinics retrieveClinicsByStopCodes(String siteCode, StopCodes stopCodes) {
        String stopCodePairsAsString = ClinicalService.getStopCodesAsString(stopCodes);

        Map<String, Object> inputParamsMap = new HashMap<>();
        inputParamsMap.put(DAOConstants.STATION, siteCode);
        inputParamsMap.put(DAOConstants.STOP_CODE_PAIRS, stopCodePairsAsString);
        inputParamsMap.put(DAOConstants.DISP_APPT_FLAG, null);
        inputParamsMap.put(DAOConstants.DIRECT_SCHED_FLAG, null);
        inputParamsMap.put(DAOConstants.CLINIC_IENS, null);

        return executeStoredProcedure(inputParamsMap);
    }

    public ClinicStoredProcedure getClinicStoredProcedure() {
        return clinicStoredProcedure;
    }

    public void setClinicStoredProcedure(ClinicStoredProcedure clinicStoredProcedure) {
        this.clinicStoredProcedure = clinicStoredProcedure;
    }

    public JdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }

    public void setJdbcTemplate(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    public String getStoredProcedure() {
        return storedProcedure;
    }

    public void setStoredProcedure(String storedProcedure) {
        this.storedProcedure = storedProcedure;
    }

    public ClinicResultSetRowMapper getClinicResultSetRowMapper() {
        return clinicResultSetRowMapper;
    }

    public void setClinicResultSetRowMapper(ClinicResultSetRowMapper clinicResultSetRowMapper) {
        this.clinicResultSetRowMapper = clinicResultSetRowMapper;
    }

    private CdwClinics filterCdwClinics(CdwClinics cdwClinics, String institutionCode, Collection<StopCode> excludedStopCodes) {
        CdwClinics filteredClinics = new CdwClinics();

        for (CdwClinic cdwClinic : cdwClinics) {
            final StopCode clinicStopCode = new StopCode(cdwClinic.getPrimaryStopCode(), cdwClinic.getSecondaryStopCode());
            if (excludedStopCodes == null || !StopCode.matchAny(excludedStopCodes, clinicStopCode)) {
                if (cdwClinic.getInstitutionCode().equals(institutionCode)) {
                    filteredClinics.add(cdwClinic);
                }
            }
        }

        return filteredClinics;
    }
}
